//	TorusGames2DWordSearch.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesSound.h"
#include <math.h>


//	When testing various word search puzzle data sets,
//	let the computer solve the puzzles to save time.
//#define WORD_SEARCH_AUTO_SOLVE
#ifdef WORD_SEARCH_AUTO_SOLVE
#warning WORD_SEARCH_AUTO_SOLVE is enabled
#endif

#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_WORD_SEARCH	5

//	Let the lines that mark found words be a smidge thinner
//	than the grid spacing, so that when two adjacent words
//	get marked, a tiny bit of the background color shows
//	between them.
#define WORD_MARKER_THICKNESS_FACTOR					0.96875

//	How close to the center of a cell (in cell units) must a click or tap be
//	in order to start a selection rather than a scroll?
#define WORD_SEARCH_SELECTION_HIT_RADIUS				0.375

//	In the touch-based interface, when continuing a previously started selection,
//	how close (in board units) must the new touch be to the previous selection's 
//	hot spot to count as a hit?
#define WORD_SEARCH_SELECTION_CONTINUATION_HIT_RADIUS	0.1

//	In the touch-based interface, the user taps to end a selection.
//	How close must the touch point stay to the hot point in order to count as a tap?
//	[OLD VERSION:  How much accidental movement should we allow in the definition of a tap?]
#define WORD_SEARCH_TAP_TOLERANCE						0.05

//	Glyph size
#define WORDSEARCH_CHARACTER_TEXTURE_SIZE	128


//	Public functions with private names
static void		WordSearchReset(ModelData *md);
static void		WordSearchHandMoved(ModelData *md);
static bool		WordSearchDragBegin(ModelData *md, bool aRightClick);
static void		WordSearchDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void		WordSearchDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void		WordSearchSimulationUpdate(ModelData *md);
static void		WordSearchRefreshMessage(ModelData *md);

//	Private functions
static bool		ScratchWordFromList(ModelData *md, unsigned int aStartH, unsigned int aStartV, signed int aDeltaH, signed int aDeltaV);
static void		NormalizeCellCoordinates(ModelData *md, signed int *aCellH, signed int *aCellV);
static bool		SameWord(Char16 aWordA[MAX_WORDSEARCH_WORD_LENGTH + 1], Char16 aWordB[MAX_WORDSEARCH_WORD_LENGTH + 1]);
static void		RemoveSpacesHyphensAndPunctuation(Char16 aWord[MAX_WORDSEARCH_WORD_LENGTH + 1]);
static void		GetHotPoint(ModelData *md, Placement2D *aHotPoint);
static void		HSVtoRGB(double aHue, double aSaturation, double aValue, float anRGBAColor[4]);


void WordSearch2DSetUp(ModelData *md)
{
	const Char16	*theLanguageCode;
	unsigned int	h,
					v;

	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &WordSearchReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= &WordSearchHandMoved;
	md->itsGame2DDragBegin				= &WordSearchDragBegin;
	md->itsGame2DDragObject				= &WordSearchDragObject;
	md->itsGame2DDragEnd				= &WordSearchDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &WordSearchSimulationUpdate;
	md->itsGameRefreshMessage			= &WordSearchRefreshMessage;

	//	Initialize variables for robust error handling.
	md->itsGameOf.WordSearch2D.itsPuzzleSize	= 0;
	
	//	Note the initial language.
	theLanguageCode = GetCurrentLanguage();
	md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[0] = theLanguageCode[0];
	md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[1] = theLanguageCode[1];
	md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[2] = 0;

	//	Start at the beginning of each puzzle list.
	md->itsGameOf.WordSearch2D.itsIndexTorus = 0;
	md->itsGameOf.WordSearch2D.itsIndexKlein = 0;

	//	Set up an empty board to ensure safe texture initialization.
	for (h = 0; h < MAX_WORDSEARCH_SIZE; h++)
		for (v = 0; v < MAX_WORDSEARCH_SIZE; v++)
			md->itsGameOf.WordSearch2D.itsBoard[h][v] = ' ';

	//	No selection is pending.
	md->itsGameOf.WordSearch2D.itsWordSelectionIsPending			= false;
	md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag	= false;
	
	//	Reset the word search puzzle.
	WordSearchReset(md);
}


const Char16 *ChooseWordSearchCellFont(void)
{
	//	Choose a font that looks good and is available on the host platform.
#if TARGET_OS_OSX
	if (IsCurrentLanguage(u"ja"))
	{
		//	"Hiragino Kaku Gothic Pro W3" and "Hiragino Kaku Gothic Pro W6"
		//		are gothic fonts of lighter and heavier line weights.
		//	"Hiragino Mincho Pro W3" and "Hiragino Mincho Pro W6"
		//		are mincho fonts of lighter and heavier line weights.
		//	"Arial Unicode MS" is a gothic font with a fairly heavy line weight.
		//
		//	Tatsu says
		//
		//		I do not think Mincho is a good match to the word search puzzle.
		//		I find it difficult to scan the grid for words.  
		//		You should stick with Kaku-Gothic.  W3 looks good, W6 a bit too thick.
		//
		return u"Hiragino Kaku Gothic Pro W3";
	}
	else
	if (IsCurrentLanguage(u"ko"))
	{
		//	"AppleMyungjo Regular" is easy to read,
		//		yet still has a human touch.
		//	"AppleGothic Regular" is very easy to read,
		//		but lacks AppleMyungjo's human touch.
		//	"GungSeo Regular" (궁서) has personality,
		//		but the characters seems a little small,
		//		while the spacing between them seems a little large.
		//	"PilGi Regular" looks like handwriting.
		//		OK, but maybe a little hard to read.
		//	"Arial Unicode MS" is like "AppleGothic Regular"
		//		but slightly bolder.
		//
		//	Note that macOS is fussy about font names:
		//	"GungSeo Regular" works while "GungSeo" fails, yet
		//	"Arial Unicode MS Regular" fails while "Arial Unicode MS" works.
		//	Go figure.
		//
		return u"AppleGothic Regular";
	}
	else
	if (IsCurrentLanguage(u"zs"))
	{
		//	For options, the page
		//		http://www.yale.edu/chinesemac/pages/fonts.html
		//	contains a chart "Fonts via Apple" that says
		//	which fonts are included in which versions of OS.
		return u"Heiti SC";
	}
	else
	if (IsCurrentLanguage(u"zt"))
	{
		//	For options, see comment immediately above.
		return u"Heiti TC";
	}
	else
	{
		return u"Helvetica";
	}
#elif TARGET_OS_IOS
	//	The page
	//
	//		http://iosfonts.com
	//
	//	shows which fonts are included in which versions of iOS.
	//
	if (IsCurrentLanguage(u"ja"))
	{
		//	"HiraKakuProN-W3" and "HiraKakuProN-W6"
		//		are gothic fonts of lighter and heavier line weights.
		//	"HiraMinProN-W3" and "HiraMinProN-W6"
		//		are mincho fonts of lighter and heavier line weights,
		//		but are available on iPad only, not on iPhone.
		//
		//	Tatsu says
		//
		//		I do not think Mincho is a good match to the word search puzzle.
		//		I find it difficult to scan the grid for words.  
		//		You should stick with Kaku-Gothic.  W3 looks good, W6 a bit too thick.
		//
		return u"HiraKakuProN-W3";
	}
	else
	if (IsCurrentLanguage(u"ko"))
	{
		//	"AppleGothic" is the only font from the above-mentioned lists
		//		that I recognize as supporting Korean.
		return u"AppleGothic";
	}
	else
	if (IsCurrentLanguage(u"zs"))
	{
		//	Reasonable choices are "STHeitiSC-Light" or ""STHeitiSC-Medium".
		return u"STHeitiSC-Light";
	}
	else
	if (IsCurrentLanguage(u"zt"))
	{
		//	For options, see comment immediately above.
		//	For consistency, use the "TC" variant of the "SC" font chosen above.
		return u"STHeitiTC-Light";
	}
	else
	{
		return u"Helvetica";
	}
#else
#error Word Search cell font not specified
#endif
}


static void WordSearchReset(ModelData *md)
{
	const Char16	*theLanguageCode;
	Char16			theRawPuzzleData[1024],
					*r;	//	the read position
	unsigned int	theNumPuzzles,
					*theIndex,
					h,
					v,
					i,
					j;
#ifdef WORD_SEARCH_AUTO_SOLVE
	signed int		dh,
					dv;
	unsigned int	theLength;
#endif
	
	//	Cancel any pending selection.
	md->itsGameOf.WordSearch2D.itsWordSelectionIsPending = false;
	
	//	If the user changed languages, reset the puzzle indices to 0 
	//	so the user will get the easiest puzzle in the new language.
	if ( ! IsCurrentLanguage(md->itsGameOf.WordSearch2D.itsPreviousLanguageCode) )
	{
		md->itsGameOf.WordSearch2D.itsIndexTorus = 0;
		md->itsGameOf.WordSearch2D.itsIndexKlein = 0;

		theLanguageCode = GetCurrentLanguage();
		md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[0] = theLanguageCode[0];
		md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[1] = theLanguageCode[1];
		md->itsGameOf.WordSearch2D.itsPreviousLanguageCode[2] = 0;
	}

	//	Read the raw puzzle data as a string resource.
	theNumPuzzles	= GetNumPuzzles(Game2DWordSearch, md->itsTopology);
	theIndex		= md->itsTopology == Topology2DTorus ?
						&md->itsGameOf.WordSearch2D.itsIndexTorus :
						&md->itsGameOf.WordSearch2D.itsIndexKlein;
	*theIndex		%= theNumPuzzles;	//	unnecessary but safe
#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
	//	Get the desired puzzle for the screenshot.
	GEOMETRY_GAMES_ASSERT(md->itsTopology == Topology2DTorus, "Torus screenshots expected");
	if (IsCurrentLanguage(u"de")) *theIndex = 0;
	if (IsCurrentLanguage(u"el")) *theIndex = 0;
	if (IsCurrentLanguage(u"en")) *theIndex = 0;
	if (IsCurrentLanguage(u"es")) *theIndex = 0;
	if (IsCurrentLanguage(u"fi")) *theIndex = 0;
	if (IsCurrentLanguage(u"fr")) *theIndex = 0;
	if (IsCurrentLanguage(u"it")) *theIndex = 0;
	if (IsCurrentLanguage(u"ja")) *theIndex = 0;
	if (IsCurrentLanguage(u"ko")) *theIndex = 0;
	if (IsCurrentLanguage(u"nl")) *theIndex = 0;
	if (IsCurrentLanguage(u"pt")) *theIndex = 0;
	if (IsCurrentLanguage(u"ru")) *theIndex = 0;
	if (IsCurrentLanguage(u"vi")) *theIndex = 0;
	if (IsCurrentLanguage(u"zs")) *theIndex = 0;
	if (IsCurrentLanguage(u"zt")) *theIndex = 0;
#endif
	GetRawPuzzleData(	Game2DWordSearch,
						md->itsTopology,
						*theIndex,
						theRawPuzzleData,
						BUFFER_LENGTH(theRawPuzzleData));

	//	Increment the index for next time.
	(*theIndex)++;
	*theIndex %= theNumPuzzles;

	//	Initialize a "read pointer".
	r = theRawPuzzleData;

	//	Confirm the leading 'T' or 'K'.
	if (*r++ != (md->itsTopology == Topology2DTorus ? 'T' : 'K'))
		goto WordSearchResetError;

	//	Parse the puzzle size.
	if (*r >= '1' && *r <= '9')
	{
		md->itsGameOf.WordSearch2D.itsPuzzleSize = *r++ - '0';
		if (*r >= '0' && *r <= '9')
		{
			md->itsGameOf.WordSearch2D.itsPuzzleSize *= 10;
			md->itsGameOf.WordSearch2D.itsPuzzleSize += *r++ - '0';
		}

		if (*r++ != '/')
			goto WordSearchResetError;
	}
	else
		goto WordSearchResetError;
	if (md->itsGameOf.WordSearch2D.itsPuzzleSize <= 0
	 || md->itsGameOf.WordSearch2D.itsPuzzleSize > MAX_WORDSEARCH_SIZE)
		goto WordSearchResetError;

	//	We've finished reading the header information.
	if (*r++ != '/')
		goto WordSearchResetError;

#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
	//	For some languages, the screenshot looks better reflected.
	bool	theReflectH = false,
			theReflectV = false;
	GEOMETRY_GAMES_ASSERT(md->itsTopology == Topology2DTorus, "Torus screenshots expected");
	if (IsCurrentLanguage(u"de")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"el")) { theReflectH = true;  theReflectV = false; }
	if (IsCurrentLanguage(u"en")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"es")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"fi")) { theReflectH = true;  theReflectV = true;  }
	if (IsCurrentLanguage(u"fr")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"it")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"ja")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"ko")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"nl")) { theReflectH = true;  theReflectV = false; }
	if (IsCurrentLanguage(u"pt")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"ru")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"vi")) { theReflectH = true;  theReflectV = true;  }
	if (IsCurrentLanguage(u"zs")) { theReflectH = false; theReflectV = false; }
	if (IsCurrentLanguage(u"zt")) { theReflectH = false; theReflectV = false; }
#endif

	//	Read the puzzle board and set all flips to false.
	//
	//	Note #1:  The internal v-coordinate runs bottom-to-top,
	//	while raw puzzle data runs top-to-bottom.
	//
	//	Note #2:  The internal coordinates are ordered (h,v),
	//	so the outer loop iterates over v, even though v is
	//	the second coordinate.
	for (v = md->itsGameOf.WordSearch2D.itsPuzzleSize; v-- > 0; )
	{
		for (h = 0; h < md->itsGameOf.WordSearch2D.itsPuzzleSize; h++)
		{
#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
			md->itsGameOf.WordSearch2D.itsBoard
					[theReflectH ? (md->itsGameOf.WordSearch2D.itsPuzzleSize - 1) - h : h]
					[theReflectV ? (md->itsGameOf.WordSearch2D.itsPuzzleSize - 1) - v : v]
				= *r++;
#else
			md->itsGameOf.WordSearch2D.itsBoard[h][v] = *r++;
#endif

			if (md->itsTopology == Topology2DTorus)
				md->itsGameOf.WordSearch2D.itsBoardFlips[h][v] = false;
			else
				md->itsGameOf.WordSearch2D.itsBoardFlips[h][v] = RandomBoolean();
		}

		if (*r++ != '/')
			goto WordSearchResetError;
	}
	if (*r++ != '/')
		goto WordSearchResetError;

	//	Read the words to be found.
	for (i = 0; i < MAX_WORDSEARCH_NUM_WORDS; i++)
	{
		if (*r != '/')
		{
			for (j = 0; j < MAX_WORDSEARCH_WORD_LENGTH + 1; j++)
			{
				if (*r == 0)
					goto WordSearchResetError;

				if (*r != '/')
					md->itsGameOf.WordSearch2D.itsWords[i][j] = *r++;
				else
					md->itsGameOf.WordSearch2D.itsWords[i][j] = 0;
			}
			if (md->itsGameOf.WordSearch2D.itsWords[i][MAX_WORDSEARCH_WORD_LENGTH] != 0)
				goto WordSearchResetError;
			r++;
		}
		else
		{
			for (j = 0; j < MAX_WORDSEARCH_WORD_LENGTH + 1; j++)
				md->itsGameOf.WordSearch2D.itsWords[i][j] = 0;
		}
	}
	if (*r != '/')
		goto WordSearchResetError;

	//	Display the current word List.
	WordSearchRefreshMessage(md);

	//	No lines have been found yet.
	md->itsGameOf.WordSearch2D.itsNumLinesFound = 0;

	//	The game is not over.
	md->itsGameIsOver	= false;
	md->itsFlashFlag	= false;

	//	Abort any pending simulation.
	SimulationEnd(md);

#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)

//	Define a macro MW to "mark word".
//	Colors are in linear Display P3.
#define MW(h,v,dh,dv,r,g,b,a)										\
		{															\
			WordSearchLine	*theLine;								\
			theLine = &md->itsGameOf.WordSearch2D.itsLines			\
				[md->itsGameOf.WordSearch2D.itsNumLinesFound];		\
			theLine->itsStartH = h;									\
			theLine->itsStartV = v;									\
			theLine->itsDeltaH = dh;								\
			theLine->itsDeltaV = dv;								\
			theLine->itsColor[0] = r;								\
			theLine->itsColor[1] = g;								\
			theLine->itsColor[2] = b;								\
			theLine->itsColor[3] = a;								\
			md->itsGameOf.WordSearch2D.itsNumLinesFound++;			\
																	\
			if (IsCurrentLanguage(u"ko")							\
			 || IsCurrentLanguage(u"zs")							\
			 || IsCurrentLanguage(u"zt"))							\
			{														\
				/*	The screenshot looks best with				*/	\
				/*	a complete list of "solar terms" in Korean	*/	\
				/*	and a complete poem in Chinese,				*/	\
				/*	so don't delete the marked words			*/	\
				/*	in those languages.							*/	\
			}														\
			else	/*	language other than Korean or Chinese	*/	\
			{														\
				ScratchWordFromList(md,  h,    v,   +dh, +dv);		\
				ScratchWordFromList(md, h+dh, v+dv, -dh, -dv);		\
			}														\
		}

//	Define a macro SSP to "set scroll position".
#define SSP(h,v)							\
		{									\
			md->itsOffset.itsH = h;			\
			md->itsOffset.itsV = v;			\
			md->itsOffset.itsFlip = false;	\
		}

#endif	//	defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)

#ifdef MAKE_GAME_CHOICE_ICONS

	if (IsCurrentLanguage(u"ru"))
	{
		MW(5, 2, +4,  0, 1.0, 0.0, 0.5, 1.0)	//	бизон
		MW(1, 3, -3, -3, 0.0, 0.5, 1.0, 1.0)	//	лисица (first four letters only)
		
		SSP(+1.0/6.0,  0.0/6.0)
	}
	else
	{
		SSP( 0.0/6.0,  0.0/6.0)	//	ResetScrolling() doesn't reset when MAKE_GAME_CHOICE_ICONS is #defined,
								//		so we should reset the scrolling here.
	}

#endif	//	MAKE_GAME_CHOICE_ICONS

#ifdef GAME_CONTENT_FOR_SCREENSHOT

	if (IsCurrentLanguage(u"de"))
	{
		MW(4, 3, +5, -5, 1.0, 0.0, 0.5, 1.0)	//	Pavian
		MW(0, 0, +5,  0, 0.0, 0.5, 1.0, 1.0)	//	Ziesel
		
		SSP(+2.0/6.0,  0.0/6.0)
	}
	if (IsCurrentLanguage(u"el"))
	{
		MW(5, 5, +4,  0, 1.0, 0.0, 0.5, 1.0)	//	Αθηνά
		MW(4, 2, -4, -4, 0.0, 0.5, 1.0, 1.0)	//	Ηρμής
		
		SSP(+3.0/6.0, -1.0/6.0)
	}
	if (IsCurrentLanguage(u"en"))
	{
		MW(0, 3, +5, -5, 1.0, 0.0, 0.5, 1.0)	//	jaguar
		MW(0, 2, +2,  0, 0.0, 0.5, 1.0, 1.0)	//	cat
		
		SSP( 0.0/6.0,  0.0/6.0)
	}
	if (IsCurrentLanguage(u"es"))
	{
		MW(2, 3,  0, -5, 1.0, 0.0, 0.5, 1.0)	//	castor
		MW(4, 3, -3,  0, 0.0, 0.5, 1.0, 1.0)	//	alce
		
		SSP( 0.0/6.0,  0.0/6.0)
	}
	if (IsCurrentLanguage(u"fi"))
	{
		MW(7, 1, +6,  0, 1.0, 0.0, 0.5, 1.0)	//	Tampere
		MW(5, 2,  0, -7, 0.0, 0.5, 1.0, 1.0)	//	Helsinki
		
		SSP(-2.0/8.0, +3.0/8.0)
	}
	if (IsCurrentLanguage(u"fr"))
	{
		MW(4, 2,  0, -5, 1.0, 0.0, 0.5, 1.0)	//	béluga
		MW(4, 0, +3, -3, 0.0, 0.5, 1.0, 1.0)	//	loup
		
		SSP(+3.0/6.0, -2.0/6.0)
	}
	if (IsCurrentLanguage(u"it"))
	{
		MW(1, 5,  0, -5, 1.0, 0.0, 0.5, 1.0)	//	pecora
		MW(3, 1, -4, -4, 0.0, 0.5, 1.0, 1.0)	//	lepre
		
		SSP(+1.0/6.0, -2.0/6.0)
	}
	if (IsCurrentLanguage(u"ja"))
	{
		MW(0, 5, -5, -5, 1.0, 0.0, 0.5, 1.0)	//	ほっかいどう
		MW(5, 2, +3, -3, 0.0, 0.5, 1.0, 1.0)	//	きょうと
		
		SSP(+3.0/6.0,  0.0/6.0)
	}
	if (IsCurrentLanguage(u"ko"))
	{
		MW(2, 2,  0, -1, 1.0, 0.0, 0.5, 1.0)	//	대서
		MW(1, 1,  0, -1, 0.0, 0.5, 1.0, 1.0)	//	소설
		MW(2, 2, +1,  0, 1.0, 1.0, 0.5, 1.0)	//	대설
		
		SSP( 0.0/6.0, -1.0/6.0)
	}
	if (IsCurrentLanguage(u"nl"))
	{
		MW(0, 1, +5, -5, 1.0, 0.0, 0.5, 1.0)	//	Leiden
		MW(6, 4, +7,  0, 0.0, 0.5, 1.0, 1.0)	//	Enschede
		
		SSP(+2.0/7.0, +1.0/7.0)
	}
	if (IsCurrentLanguage(u"pt"))
	{
		MW(1, 3,  0, -5, 1.0, 0.0, 0.5, 1.0)	//	láparo
		MW(4, 1, +3, -3, 0.0, 0.5, 1.0, 1.0)	//	lobo
		
		SSP(+2.0/6.0,  0.0/6.0)
	}
	if (IsCurrentLanguage(u"ru"))
	{
		MW(1, 4, +5,  0, 1.0, 0.0, 0.5, 1.0)	//	павиан
		MW(4, 5,  0, -4, 0.0, 0.5, 1.0, 1.0)	//	жираф
		
		SSP( 0.0/6.0, -2.0/6.0)
	}
	if (IsCurrentLanguage(u"vi"))
	{
		MW(4, 3,  0, -3, 1.0, 0.0, 0.5, 1.0)	//	ngưu
		MW(3, 3, +3, -3, 0.0, 0.5, 1.0, 1.0)	//	ngựa
		
		SSP( 0.0/6.0, -1.0/6.0)
	}
	if (IsCurrentLanguage(u"zs"))
	{
		MW(5, 5,  0, -4, 1.0, 0.0, 0.5, 1.0)	//	当春乃发生
		MW(0, 5, +4,  0, 0.0, 0.5, 1.0, 1.0)	//	好雨知时节
		
		SSP(+2.0/6.0, -2.0/6.0)
	}
	if (IsCurrentLanguage(u"zt"))
	{
		MW(5, 5,  0, -4, 1.0, 0.0, 0.5, 1.0)	//	當春乃發生
		MW(0, 5, +4,  0, 0.0, 0.5, 1.0, 1.0)	//	好雨知時節
		
		SSP(+2.0/6.0, -2.0/6.0)
	}

#endif	//	GAME_CONTENT_FOR_SCREENSHOT

#ifdef WORD_SEARCH_AUTO_SOLVE

	//	Note:  Use auto-solve to find only words of length at most itsPuzzleSize.
	//	Find longer words by hand to make sure they respect conditions
	//	about not extending more than half a board width beyond
	//	the fundamental square.

	for (h = 0; h < md->itsGameOf.WordSearch2D.itsPuzzleSize; h++)
	{
		for (v = 0; v < md->itsGameOf.WordSearch2D.itsPuzzleSize; v++)
		{
			for (dh = -1; dh <= +1; dh++)
			{
				for (dv = -1; dv <= +1; dv++)
				{
					if (dh != 0 || dv != 0)
					{
						for (theLength = 1; theLength <= md->itsGameOf.WordSearch2D.itsPuzzleSize - 1; theLength++)	//	see note above
						{
							md->itsGameOf.WordSearch2D.itsStartH = h;
							md->itsGameOf.WordSearch2D.itsStartV = v;
							md->itsGameOf.WordSearch2D.itsDeltaH = dh * (signed int) theLength;
							md->itsGameOf.WordSearch2D.itsDeltaV = dv * (signed int) theLength;
							
							//	The R, G and B components should be premultiplied by the alpha component,
							//	but as long as the alpha component is 1.0 there's no need.
							HSVtoRGB(
								RandomFloat(),	//	hue
								1.0,			//	saturation
								1.0,			//	value
								md->itsGameOf.WordSearch2D.itsColor);
							
							WordSearchDragEnd(md, 0.0, false);
						}
					}
				}
			}
		}
	}
#endif

	return;

WordSearchResetError:

	GEOMETRY_GAMES_ABORT("Invalid Word Search puzzle data.");

	return;
}


static void WordSearchRefreshMessage(ModelData *md)	//	display the current word list
{
	Char16			theWordList[STATUS_MESSAGE_BUFFER_SIZE];
	bool			theFirstWordFlag;
	unsigned int	i;

	const ColorP3Linear	theWordListColor		= {0.0, 0.0, 0.0, 1.0};
	const Char16		*theNewlineCharacter	= u"\n";	//	would be u"\r\n" on Windows

	theWordList[0] = 0;
	
	theFirstWordFlag = true;

	for (i = 0; i < MAX_WORDSEARCH_NUM_WORDS; i++)
	{
		if (md->itsGameOf.WordSearch2D.itsWords[i][0] != 0)
		{
			//	Insert a comma and/or newline?
			if (theFirstWordFlag)
			{
				//	The first word never needs a preceding comma or newline.
				
				//	The next word we find won't be the first word.
				theFirstWordFlag = false;
			}
			else	//	The current word is not the first word.
			{
				if (IsCurrentLanguage(u"zs") || IsCurrentLanguage(u"zt"))
				{
					//	The Chinese poems already have punctuation,
					//	so no commas are needed.
					//
					//	Each line comes in two halves, for example
					//
					//		危楼高百尺，手可摘星辰。
					//		不敢高声语，恐惊天上人。
					//
					//	The word search puzzles treat each half as a "word".
					//
					if (md->itsChinesePoemsWantColumnFormatting)
					{
						//	In landscape orienation, the word list area is tall and narrow.
						//	Put a newline before each "word", so the poem looks like
						//
						//			危楼高百尺，
						//			手可摘星辰。
						//			不敢高声语，
						//			恐惊天上人。
						//
						Strcat16(	theWordList,
									BUFFER_LENGTH(theWordList),
									theNewlineCharacter);
					}
					else
					{
						//	In portrait orienation on a mobile device
						//	(or any orientation on a Mac or Windows computer),
						//	the word list area is short and wide.
						//	Format the poem as
						//
						//		危楼高百尺，手可摘星辰。
						//		不敢高声语，恐惊天上人。
						//
						if (i & 0x1)	//	odd-numbered words (second half of a line)
						{
							//	An odd-numbered "word" (3, 5, …) needs a preceding newline
							//	iff its corresponding even-numbered word (2, 4, …)
							//	has already been deleted from the list.
							//
							if (md->itsGameOf.WordSearch2D.itsWords[i-1][0] == 0)
							{
								Strcat16(	theWordList,
											BUFFER_LENGTH(theWordList),
											theNewlineCharacter);
							}
						}
						else			//	even-numbered words (first half of a line)
						{
							//	An even-numbered "word" (2, 4, …)
							//	always needs a preceding newline.
							Strcat16(	theWordList,
										BUFFER_LENGTH(theWordList),
										theNewlineCharacter);
						}
					}
				}
				else
				if (IsCurrentLanguage(u"ja"))
				{
					//	In Japanese, insert an ideographic comma
					//	(but no extra space) before every word
					//	except the first one.
					Strcat16(	theWordList,
								BUFFER_LENGTH(theWordList),
								u"、");
				}
				else
				{
					//	In all languages except Chinese and Japanese,
					//	insert a comma and a space before every word
					//	except the first one.
					Strcat16(	theWordList,
								BUFFER_LENGTH(theWordList),
								u", ");
				}
			}

			Strcat16(	theWordList,
						BUFFER_LENGTH(theWordList),
						md->itsGameOf.WordSearch2D.itsWords[i]);
		}
	}

	SetTorusGamesStatusMessage(theWordList, theWordListColor, Game2DWordSearch);
}


static void WordSearchHandMoved(ModelData *md)
{
	double			theHandH,
					theHandV;
	unsigned int	h,
					v;

	//	Convert hand coordinates from [-0.5, +0.5] to [0.0, PuzzleSize].
	theHandH = md->itsGameOf.WordSearch2D.itsPuzzleSize*(md->its2DHandPlacement.itsH + 0.5);
	theHandV = md->itsGameOf.WordSearch2D.itsPuzzleSize*(md->its2DHandPlacement.itsV + 0.5);

	//	Find the integer coordinates (h,v) of the hit cell.
	h = (unsigned int)(signed int) floor(theHandH);	//	-1 would map to 0xFFFFFFFF and get caught below.
	v = (unsigned int)(signed int) floor(theHandV);

	//	Exclude values on the boundary and
	//	protect against unexpected errors.
	if (h >= md->itsGameOf.WordSearch2D.itsPuzzleSize
	 || v >= md->itsGameOf.WordSearch2D.itsPuzzleSize)
		return;

	//	The orientation of the symbol relative to the fundamental
	//	domain should match the current orientation of the hand
	//	relative to the fundamental domain.
	md->itsGameOf.WordSearch2D.itsBoardFlips[h][v] = md->its2DHandPlacement.itsFlip;
}


static bool WordSearchDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
	Placement2D	theHotPoint;
#endif
	double		theHandH,
				theHandV;

	UNUSED_PARAMETER(aRightClick);

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	The touch-based and mouse-based versions of torus word search handle drags differently.
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE

	//	Touch-based interface
	
	//	If a selection is already in progress...
	if (md->itsGameOf.WordSearch2D.itsWordSelectionIsPending)
	{
		//	Let WordSearchDragEnd() know that a word selection was present
		//	even before this drag began.
		md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag = false;

		//	Locate the hot point.
		GetHotPoint(md, &theHotPoint);

		//	Typically, the drag is reversed if and only if
		//	theHotPoint.itsFlip disagrees with its2DHandPlacement.itsFlip.
		//	The exceptional case -- when theHotPoint sits near the top of the fundamental domain
		//	while its2DHandPlacement sits near the bottom, or vice versa -- gets corrected for
		//	within the call to Shortest2DDistance(), immediately below.
		md->its2DDragIsReversed = (theHotPoint.itsFlip != md->its2DHandPlacement.itsFlip);

		//	If the user has touched theHotPoint...
		if (Shortest2DDistance(
					md->its2DHandPlacement.itsH,
					md->its2DHandPlacement.itsV,
					theHotPoint.itsH,
					theHotPoint.itsV,
					md->itsTopology,
					&md->its2DDragIsReversed)
				< WORD_SEARCH_SELECTION_CONTINUATION_HIT_RADIUS)
		{
			//	...then let the pending selection continue.
			//	That is, start a new drag which will be part of the same pending selection.
			md->itsGameOf.WordSearch2D.itsDragStartH	= md->its2DHandPlacement.itsH;
			md->itsGameOf.WordSearch2D.itsDragStartV	= md->its2DHandPlacement.itsV;
			md->itsGameOf.WordSearch2D.itsDragWasTap	= true;

			//	All done!
			return true;
		}
		else	//	But if the user touches elsewhere,
		{
			//	treat the touch as a scroll.
			//	The pre-existing pending selection remains valid but suspended,
			//	awaiting future touches for its final completion.
			return false;
		}
	}
	
	//	No selection is in progress, so let the code below start a new one.

#else
	//	Mouse-based interface
	//	Selections are never left in suspension, so no extra code is needed here.
#endif
	
	//	If we return false for any reason, then no selection will be pending.
	md->itsGameOf.WordSearch2D.itsWordSelectionIsPending			= false;
	md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag	= false;

	//	Convert hand coordinates from [-0.5, +0.5] to [0.0, PuzzleSize].
	theHandH = md->itsGameOf.WordSearch2D.itsPuzzleSize*(md->its2DHandPlacement.itsH + 0.5);
	theHandV = md->itsGameOf.WordSearch2D.itsPuzzleSize*(md->its2DHandPlacement.itsV + 0.5);

	//	Find the integer coordinates (h,v) of the hit cell.
	md->itsGameOf.WordSearch2D.itsStartH = (unsigned int)(signed int) floor(theHandH);	//	-1 would map to 0xFFFFFFFF and get caught below.
	md->itsGameOf.WordSearch2D.itsStartV = (unsigned int)(signed int) floor(theHandV);

	//	Exclude values on the boundary and protect against unexpected errors.
	if (md->itsGameOf.WordSearch2D.itsStartH >= md->itsGameOf.WordSearch2D.itsPuzzleSize
	 || md->itsGameOf.WordSearch2D.itsStartV >= md->itsGameOf.WordSearch2D.itsPuzzleSize)
		return false;

	//	Are we near the center of the hit cell?
	//	If not, treat this hit as a scroll.
	md->itsGameOf.WordSearch2D.itsDeltaH = theHandH - (md->itsGameOf.WordSearch2D.itsStartH + 0.5);
	md->itsGameOf.WordSearch2D.itsDeltaV = theHandV - (md->itsGameOf.WordSearch2D.itsStartV + 0.5);
	if (md->itsGameOf.WordSearch2D.itsDeltaH * md->itsGameOf.WordSearch2D.itsDeltaH
	  + md->itsGameOf.WordSearch2D.itsDeltaV * md->itsGameOf.WordSearch2D.itsDeltaV
	  >  WORD_SEARCH_SELECTION_HIT_RADIUS  *  WORD_SEARCH_SELECTION_HIT_RADIUS )
		return false;

	//	The orientation of the symbol relative to the fundamental
	//	domain should match the current orientation of the hand
	//	relative to the fundamental domain.
	md->itsGameOf.WordSearch2D.itsBoardFlips[md->itsGameOf.WordSearch2D.itsStartH][md->itsGameOf.WordSearch2D.itsStartV] = md->its2DHandPlacement.itsFlip;

	//	Normally aDragIsReversed specifies the relative parity
	//	of the hand and object being dragged.  In the present case,
	//	however, we're not so much dragging an object as making
	//	a mark on the board itself.  So let its2DDragIsReversed
	//	record the parity of the hand relative to the board, 
	//	at the time of the MouseDown.
	md->its2DDragIsReversed = md->its2DHandPlacement.itsFlip;	//	initially "false" in touch-based UI

	//	Choose a random opaque color.
	//
	//	The R, G and B components should be premultiplied by the alpha component,
	//	but as long as the alpha component is 1.0 there's no need.
	HSVtoRGB(
		RandomFloat(),	//	hue
		1.0,			//	saturation
		1.0,			//	value
		md->itsGameOf.WordSearch2D.itsColor);

	//	A new word selection has begun.
	md->itsGameOf.WordSearch2D.itsWordSelectionIsPending			= true;
	md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag	= true;
	
	//	The first drag of that (possibly multi-drag) word selection has begun
	//	(relevant only in touch interface).
	md->itsGameOf.WordSearch2D.itsDragStartH = ((md->itsGameOf.WordSearch2D.itsStartH + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
	md->itsGameOf.WordSearch2D.itsDragStartV = ((md->itsGameOf.WordSearch2D.itsStartH + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
	md->itsGameOf.WordSearch2D.itsDragWasTap = false;	//	A selection's first drag never counts as a terminating tap.

	return true;
}


static void WordSearchDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	double	theNewDeltaH,
			theNewDeltaV,
			theStartH,
			theStartV,
			theFinishH,
			theFinishV;

	//	Before moving anything, check how big the delta would be.
	//	Visualized in the board's universal cover, the delta stays
	//	in the coordinates of the original image of the fundamental domain.
	//	(Accumulating numerical error isn't a problem, but if it were we could
	//	adjust the delta to stay synchronized with the hand.)
	theNewDeltaH = md->itsGameOf.WordSearch2D.itsDeltaH
				+ md->itsGameOf.WordSearch2D.itsPuzzleSize *
				(md->its2DDragIsReversed ? -aHandLocalDeltaH : aHandLocalDeltaH);
	theNewDeltaV = md->itsGameOf.WordSearch2D.itsDeltaV
				+ md->itsGameOf.WordSearch2D.itsPuzzleSize * aHandLocalDeltaV;

	//	The TorusGames rendering algorithm will draw the word marker
	//	correctly iff it does not extend more than 0.5 board widths
	//	beyond the image of fundamental domain containing the center
	//	of the marker line.  A little thought confirms that the following
	//	code is both correct and optimal.  Note that if we cannot
	//	validly extend the marker line, we simply return without
	//	updating anything (not even the hand cursor, which the user
	//	will perceive as blocked).
	theStartH	= ((md->itsGameOf.WordSearch2D.itsStartH + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
	theStartV	= ((md->itsGameOf.WordSearch2D.itsStartV + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
	theFinishH	= theStartH + (theNewDeltaH / md->itsGameOf.WordSearch2D.itsPuzzleSize);
	theFinishV	= theStartV + (theNewDeltaV / md->itsGameOf.WordSearch2D.itsPuzzleSize);
	if (theFinishH > (theStartH > 0.0 ? +2.0 : +1.0)
	 || theFinishH < (theStartH < 0.0 ? -2.0 : -1.0)
	 || theFinishV > (theStartV > 0.0 ? +2.0 : +1.0)
	 || theFinishV < (theStartV < 0.0 ? -2.0 : -1.0))
		return;

	//	Update the delta.
	md->itsGameOf.WordSearch2D.itsDeltaH = theNewDeltaH;
	md->itsGameOf.WordSearch2D.itsDeltaV = theNewDeltaV;

	//	Move the hand.
	md->its2DHandPlacement.itsH += md->its2DHandPlacement.itsFlip
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	md->its2DHandPlacement.itsV += aHandLocalDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);
	
	//	If the user's finger has travelled a non-trivial distance
	//	from the start of the present drag, then the drag is not a tap.
	//	This is relevant only in the touch-based interface, not the mouse-based one.
	//	In the touch-based interface, a selection may consist of several separate drags,
	//	terminated by a tap.
	if (Shortest2DDistance(
				md->itsGameOf.WordSearch2D.itsDragStartH,
				md->itsGameOf.WordSearch2D.itsDragStartV,
				md->its2DHandPlacement.itsH,
				md->its2DHandPlacement.itsV,
				md->itsTopology,
				NULL)
			> WORD_SEARCH_TAP_TOLERANCE)
	{
		md->itsGameOf.WordSearch2D.itsDragWasTap = false;
	}

	//	Flip the underlying letter if necessary.
	WordSearchHandMoved(md);
}


static void WordSearchDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	signed int		theDeltaH,
					theDeltaV;
	WordSearchLine	*theNewlyFoundLine;
	bool			theWordsRemainFlag;
	unsigned int	i;
	
	UNUSED_PARAMETER(aDragDuration);
	
	//	If a touch sequence got cancelled (perhaps because a gesture was recognized)...
	if (aTouchSequenceWasCancelled)
	{
		//	If this touch sequence created the current word selection
		//	(most likely unintentionally)...
		if (md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag)
		{
			//	...then cancel the current selection.
			md->itsGameOf.WordSearch2D.itsWordSelectionIsPending			= false;
			md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag	= false;
		}
		else
		{
			//	The current word selection was present even before this touch sequence began,
			//	so let the selection remain pending, awaiting subsequent drags.
		}
		
		//	Whether or not we just cancelled the touch, we're done handling this drag.
		return;
	}

	//	The touch-based and mouse-based versions of torus word search handle drags differently.
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE

	//	Touch-based interface
	
	//	If the selection was not a simple tap, then even though the drag is over,
	//	let the selection remain pending, awaiting subsequent drags.
	if ( ! md->itsGameOf.WordSearch2D.itsDragWasTap )
		return;

	//	A simple tap ends both the drag and the move that the drag is a part of.
	//	Continue with the code below.

#else
	//	Mouse-based interface
	//	Each selection consists of a single drag.
#endif

	//	The selection is complete.
	md->itsGameOf.WordSearch2D.itsWordSelectionIsPending			= false;
	md->itsGameOf.WordSearch2D.itsWordSelectionBeganWithCurrentDrag	= false;

	//	What integer distance did the mouse travel?
	theDeltaH = (signed int) floor(md->itsGameOf.WordSearch2D.itsDeltaH + 0.5);
	theDeltaV = (signed int) floor(md->itsGameOf.WordSearch2D.itsDeltaV + 0.5);

	//	Accept only horizontal, vertical and diagonal lines.
	if (theDeltaH != 0
	 && theDeltaV != 0
	 && ABS(theDeltaH) != ABS(theDeltaV))
	{
		return;
	}

	//	Reject invalid words.
	if
	(
		! ScratchWordFromList(	md,
								md->itsGameOf.WordSearch2D.itsStartH,
								md->itsGameOf.WordSearch2D.itsStartV,
								+theDeltaH,
								+theDeltaV)
	 &&
		! ScratchWordFromList(	md,
								md->itsGameOf.WordSearch2D.itsStartH + theDeltaH,
								md->itsGameOf.WordSearch2D.itsStartV + theDeltaV,
								-theDeltaH,
								-theDeltaV)
	)
	{
		return;
	}

	//	Just to be safe, double check that we're
	//	not accepting too many lines.
	if (md->itsGameOf.WordSearch2D.itsNumLinesFound >= MAX_WORDSEARCH_NUM_WORDS)
		return;

	//	Record the new line.
	theNewlyFoundLine = &md->itsGameOf.WordSearch2D.itsLines[md->itsGameOf.WordSearch2D.itsNumLinesFound];
	theNewlyFoundLine->itsStartH	= md->itsGameOf.WordSearch2D.itsStartH;
	theNewlyFoundLine->itsStartV	= md->itsGameOf.WordSearch2D.itsStartV;
	theNewlyFoundLine->itsDeltaH	= theDeltaH;
	theNewlyFoundLine->itsDeltaV	= theDeltaV;
	for (i = 0; i < 4; i++)
		theNewlyFoundLine->itsColor[i] = md->itsGameOf.WordSearch2D.itsColor[i];
	md->itsGameOf.WordSearch2D.itsNumLinesFound++;

	//	Still more words to be found?
	theWordsRemainFlag = false;
	for (i = 0; i < MAX_WORDSEARCH_NUM_WORDS; i++)
		if (md->itsGameOf.WordSearch2D.itsWords[i][0] != 0)
			theWordsRemainFlag = true;

	//	If no words remain, the user has won!
	if ( ! theWordsRemainFlag )
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"WordSearchComplete.mid");
		SimulationBegin(md, Simulation2DWordSearchFlash);
	}
}


static bool ScratchWordFromList(
	ModelData		*md,
	unsigned int	aStartH,
	unsigned int	aStartV,
	signed int		aDeltaH,
	signed int		aDeltaV)
{
	unsigned int	theAbsDeltaH,
					theAbsDeltaV,
					theLength,
					i,
					j;
	signed int		theCurrentH,
					theCurrentV,
					theIncrementH,
					theIncrementV,
					theNormalizedH,
					theNormalizedV;
	Char16			theCandidate[MAX_WORDSEARCH_WORD_LENGTH + 1];

	//	Re-confirm that the word is horizontal, vertical or diagonal.
	theAbsDeltaH = ABS(aDeltaH);
	theAbsDeltaV = ABS(aDeltaV);
	if (theAbsDeltaH != 0
	 && theAbsDeltaV != 0
	 && theAbsDeltaH != theAbsDeltaV)
		return false;

	//	How long is the word?
	theLength = MAX(theAbsDeltaH, theAbsDeltaV) + 1;

	//	Reject words that are too long.
	if (theLength > MAX_WORDSEARCH_WORD_LENGTH)
		return false;

	//	Assemble the candidate word, filling unused spaces with zeros.
	//	We're guaranteed to have at least one zero.

	theCurrentH = (signed int) aStartH;
	theCurrentV = (signed int) aStartV;

	theIncrementH = SIGN(aDeltaH);
	theIncrementV = SIGN(aDeltaV);

	for (i = 0; i < MAX_WORDSEARCH_WORD_LENGTH + 1; i++)
	{
		if (i < theLength)
		{
			theNormalizedH = theCurrentH;
			theNormalizedV = theCurrentV;
			NormalizeCellCoordinates(md, &theNormalizedH, &theNormalizedV);

			theCandidate[i] = md->itsGameOf.WordSearch2D.itsBoard[theNormalizedH][theNormalizedV];

			theCurrentH += theIncrementH;
			theCurrentV += theIncrementV;
		}
		else
			theCandidate[i] = 0;
	}

	//	Does the candidate appear on the word list?
	//	If so, remove it.
	for (i = 0; i < MAX_WORDSEARCH_NUM_WORDS; i++)
	{
		if (SameWord(theCandidate, md->itsGameOf.WordSearch2D.itsWords[i]))
		{
			//	Clear the word from the list.
			for (j = 0; j < MAX_WORDSEARCH_WORD_LENGTH + 1; j++)
				md->itsGameOf.WordSearch2D.itsWords[i][j] = 0;

			//	Refresh the Word List window.
			WordSearchRefreshMessage(md);

			//	Report success.
			return true;
		}
	}

	//	The candidate wasn't on the list.
	return false;
}


static void NormalizeCellCoordinates(
	ModelData	*md,
	signed int	*aCellH,
	signed int	*aCellV)
{
	while (*aCellV >= (signed int) md->itsGameOf.WordSearch2D.itsPuzzleSize)
	{
		*aCellV -= md->itsGameOf.WordSearch2D.itsPuzzleSize;
		if (md->itsTopology == Topology2DKlein)
			*aCellH = (md->itsGameOf.WordSearch2D.itsPuzzleSize - 1) - *aCellH;
	}

	while (*aCellV < 0)
	{
		*aCellV += md->itsGameOf.WordSearch2D.itsPuzzleSize;
		if (md->itsTopology == Topology2DKlein)
			*aCellH = (md->itsGameOf.WordSearch2D.itsPuzzleSize - 1) - *aCellH;
	}

	while (*aCellH >= (signed int) md->itsGameOf.WordSearch2D.itsPuzzleSize)
		*aCellH -= md->itsGameOf.WordSearch2D.itsPuzzleSize;

	while (*aCellH < 0)
		*aCellH += md->itsGameOf.WordSearch2D.itsPuzzleSize;
}


static bool SameWord(
	Char16	aWordA[MAX_WORDSEARCH_WORD_LENGTH + 1],
	Char16	aWordB[MAX_WORDSEARCH_WORD_LENGTH + 1])
{
	Char16			theWordA[MAX_WORDSEARCH_WORD_LENGTH + 1],
					theWordB[MAX_WORDSEARCH_WORD_LENGTH + 1];
	unsigned int	i;

	//	Copy the two words, making the following changes as we go along:
	//
	//		Convert to lower case
	//			(this makes a difference only in languages that
	//			make an uppercase/lowercase distinction --
	//			in other languages ToLowerCase() has no effect).
	//	
	//		Promote small kana to large kana
	//			(this makes a difference only in Japanese -- 
	//			in other languages PromoteSmallKana() has no effect).
	//
	//		Remove Greek tonos (stress-accent) and convert final 'ς' to 'σ'
	//			(this makes a difference only in Greek -- in other languages
	//			RemoveTonos() and ConvertFinalSigma() have no effect).
	//
	for (i = 0; i < MAX_WORDSEARCH_WORD_LENGTH + 1; i++)
	{
		theWordA[i] =	ConvertFinalSigma(
						RemoveTonos(
						PromoteSmallKana(
						ToLowerCase(
							aWordA[i]))));

		theWordB[i] =	ConvertFinalSigma(
						RemoveTonos(
						PromoteSmallKana(
						ToLowerCase(
							aWordB[i]))));
	}

	//	Remove spaces, hyphens and other punctuation from both words.
	RemoveSpacesHyphensAndPunctuation(theWordA);
	RemoveSpacesHyphensAndPunctuation(theWordB);

	//	Compare the lowercase, spaceless, punctuationless words.
	for (i = 0; i < MAX_WORDSEARCH_WORD_LENGTH + 1; i++)
	{
		if (theWordA[i] != theWordB[i])
			return false;

		if (theWordA[i] == 0)
			return true;
	}

	return false;	//	should never occur
}


static void RemoveSpacesHyphensAndPunctuation(Char16 aWord[MAX_WORDSEARCH_WORD_LENGTH + 1])
{
	unsigned int	r,	//	read index
					w;	//	write index

	//	Copy the word in place, omitting all spaces, hyphens
	//	and Chinese punctuation.  The Chinese punctuation
	//	is relevant only when the word list is a Chinese poem.
	w = 0;
	for (r = 0; r < MAX_WORDSEARCH_WORD_LENGTH + 1; r++)
	{
		if (aWord[r] != u' '
		 && aWord[r] != u'-'
		 && aWord[r] !=	u'，'
		 && aWord[r] !=	u'。'
		 && aWord[r] != u'？' )
		{
			aWord[w++] = aWord[r];
		}
	}

	//	If one or more spaces were omitted,
	//	fill the remainder of the string with zeros.
	while (w < MAX_WORDSEARCH_WORD_LENGTH + 1)
		aWord[w++] = 0;
}


static void WordSearchSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DWordSearchFlash:
			md->itsFlashFlag = (((unsigned int) floor(5.0 * md->itsSimulationElapsedTime)) % 2) ? true : false;
			if (md->itsSimulationElapsedTime >= 1.4)
			{
				SimulationEnd(md);
				md->itsFlashFlag = false;
			}
			break;

		default:
			break;
	}
}


unsigned int GetNum2DWordSearchBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_WORD_SEARCH;
}

void Get2DWordSearchKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DWordSearchSprites(
	unsigned int	aPuzzleSize,
	unsigned int	aNumLinesFound)
{
	return aPuzzleSize * aPuzzleSize	//	cells
			+ aNumLinesFound			//	marked words
			+ 2;						//	pending word and hot point
}

void Get2DWordSearchSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int		i,
						h,
						v,
						n,				//	puzzle size is n letters by n letters
						ell;			//	number of lines (words) already found in Word Search
	double				theCellSize;	//	1/n
	WordSearchLine		*theLine;
	WordSearchData2D	*theWordSearch;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DWordSearch,
		"Game2DWordSearch must be active");

	n			= md->itsGameOf.WordSearch2D.itsPuzzleSize;
	ell			= md->itsGameOf.WordSearch2D.itsNumLinesFound;
	theCellSize	= 1.0 / (double) n;

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DWordSearchSprites(n, ell),
		"Internal error:  wrong buffer size");

	//	cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = n*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + (0.5 + h) * theCellSize;
			aPlacementBuffer[i].itsV		= -0.5 + (0.5 + v) * theCellSize;
			aPlacementBuffer[i].itsFlip		= md->itsGameOf.WordSearch2D.itsBoardFlips[h][v];
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= theCellSize;
			aPlacementBuffer[i].itsSizeV	= theCellSize;
		}
	}
	
	//	lines for marked words
	for (i = 0; i < ell; i++)
	{
		theLine = &md->itsGameOf.WordSearch2D.itsLines[i];

		aPlacementBuffer[n*n + i].itsH		= -0.5 + ((theLine->itsStartH + 0.5) + 0.5*theLine->itsDeltaH) * theCellSize;
		aPlacementBuffer[n*n + i].itsV		= -0.5 + ((theLine->itsStartV + 0.5) + 0.5*theLine->itsDeltaV) * theCellSize;
		aPlacementBuffer[n*n + i].itsFlip	= false;
		aPlacementBuffer[n*n + i].itsAngle	= atan2(theLine->itsDeltaV, theLine->itsDeltaH);
		aPlacementBuffer[n*n + i].itsSizeH	= sqrt(theLine->itsDeltaH*theLine->itsDeltaH + theLine->itsDeltaV*theLine->itsDeltaV) * theCellSize;
		aPlacementBuffer[n*n + i].itsSizeV	= WORD_MARKER_THICKNESS_FACTOR * theCellSize;

		//	Normalize the Placement2D to put the line's center
		//	within the fundamental domain.  The Torus Games algorithm
		//	ensures that lines (and anything else) that extend
		//	at most 0.5 units beyond the fundamental domain
		//	will always get rendered correctly.
		Normalize2DPlacement(&aPlacementBuffer[n*n + i], md->itsTopology);
	}

	//	line for pending word, if any
	if (md->itsGameOf.WordSearch2D.itsWordSelectionIsPending)
	{
		theWordSearch = &md->itsGameOf.WordSearch2D;
		
		aPlacementBuffer[n*n + ell].itsH		= -0.5 + ((theWordSearch->itsStartH + 0.5) + 0.5*theWordSearch->itsDeltaH) * theCellSize;
		aPlacementBuffer[n*n + ell].itsV		= -0.5 + ((theWordSearch->itsStartV + 0.5) + 0.5*theWordSearch->itsDeltaV) * theCellSize;
		aPlacementBuffer[n*n + ell].itsFlip		= false;
		aPlacementBuffer[n*n + ell].itsAngle	= atan2(theWordSearch->itsDeltaV, theWordSearch->itsDeltaH);
		aPlacementBuffer[n*n + ell].itsSizeH	= sqrt(theWordSearch->itsDeltaH*theWordSearch->itsDeltaH + theWordSearch->itsDeltaV*theWordSearch->itsDeltaV) * theCellSize;
		aPlacementBuffer[n*n + ell].itsSizeV	= WORD_MARKER_THICKNESS_FACTOR * theCellSize;

		//	Normalize the Placement2D to put the pending line's center
		//	within the fundamental domain.  The Torus Games algorithm
		//	ensures that lines (and anything else) that extend
		//	at most 0.5 units beyond the fundamental domain
		//	will always get rendered correctly.
		Normalize2DPlacement(&aPlacementBuffer[n*n + ell + 0], md->itsTopology);
		
		//	hot point
		GetHotPoint(md, &aPlacementBuffer[n*n + ell + 1]);
	}
	else
	{
		aPlacementBuffer[n*n + ell + 0]	= gUnusedPlacement;
		aPlacementBuffer[n*n + ell + 1]	= gUnusedPlacement;
	}
}


static void GetHotPoint(
	ModelData	*md,		//	input
	Placement2D	*aHotPoint)	//	output
{
	double		theStartH,
				theStartV,
				theFinishH,
				theFinishV;

	if (md->itsGameOf.WordSearch2D.itsWordSelectionIsPending)
	{
		//	Locate the selection.
		theStartH	= ((md->itsGameOf.WordSearch2D.itsStartH + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
		theStartV	= ((md->itsGameOf.WordSearch2D.itsStartV + 0.5) / md->itsGameOf.WordSearch2D.itsPuzzleSize) - 0.5;
		theFinishH	= theStartH + (md->itsGameOf.WordSearch2D.itsDeltaH / md->itsGameOf.WordSearch2D.itsPuzzleSize);
		theFinishV	= theStartV + (md->itsGameOf.WordSearch2D.itsDeltaV / md->itsGameOf.WordSearch2D.itsPuzzleSize);

		//	Express the finishing point as a Placement2D.
		//	Conceptually, the finishing point inherits the starting point's unflipped chirality.
		aHotPoint->itsH		= theFinishH;
		aHotPoint->itsV		= theFinishV;
		aHotPoint->itsFlip	= false;
		aHotPoint->itsAngle	= 0.0;
		aHotPoint->itsSizeH	= 1.0 / md->itsGameOf.WordSearch2D.itsPuzzleSize;
		aHotPoint->itsSizeV	= 1.0 / md->itsGameOf.WordSearch2D.itsPuzzleSize;

		//	Normalize the finishing point to sit within the fundamental domain.
		//	The normalization will adjust the finishing point's chirality as required.
		Normalize2DPlacement(aHotPoint, md->itsTopology);
	}
	else
	{
		//	This case should never occur, but provide something to be safe.
		*aHotPoint = gUnusedPlacement;
	}
}


#pragma mark -
#pragma mark HSV to RGB

static void HSVtoRGB(	//	assumes alpha = 1.0
	double	aHue,			//	input
	double	aSaturation,	//	input
	double	aValue,			//	input
	float	anRGBAColor[4])	//	output
{
	double			theHue6;
	unsigned int	theSegment;
	double			theInterpolator;
	const double	*theLeftHue,
					*theRightHue;
	double			theRawColor[3],
					theDesaturatedColor[3],
					theDarkenedColor[3];
	unsigned int	i;

	static const double	theBaseColors[6][3] =
						{
							{1.000, 0.000, 0.000},	//	red
							{1.000, 1.000, 0.000},	//	yellow
							{0.000, 1.000, 0.000},	//	green
							{0.000, 1.000, 1.000},	//	cyan
							{0.000, 0.000, 1.000},	//	blue
							{1.000, 0.000, 1.000}	//	magenta
						};

	if (aHue < 0.0)
		aHue = 0.0;
	if (aHue > 1.0)
		aHue = 1.0;
	
	if (aSaturation < 0.0)
		aSaturation = 0.0;
	if (aSaturation > 1.0)
		aSaturation = 1.0;
	
	if (aValue < 0.0)
		aValue = 0.0;
	if (aValue > 1.0)
		aValue = 1.0;

	//	Divide the color circle into six equal segments, to be delimited
	//	by the colors {red, yellow, green, cyan, blue, magenta, red}.
	theHue6			= 6.0 * aHue;						//	∈ [0.0, 6.0]
	theSegment		= (unsigned int) floor(theHue6);	//	∈ {0,1,2,3,4,5,6}
	theInterpolator	= theHue6 - theSegment;

	theLeftHue	= theBaseColors[    theSegment    % 6 ];
	theRightHue	= theBaseColors[ (theSegment + 1) % 6 ];

	for (i = 0; i < 3; i++)
	{
		theRawColor[i]			= (1.0 - theInterpolator) * theLeftHue [i]
								+     theInterpolator     * theRightHue[i];
		
		theDesaturatedColor[i]	=     aSaturation     * theRawColor[i]
								+ (1.0 - aSaturation) *     1.0;
		
		theDarkenedColor[i]		= aValue * theDesaturatedColor[i];
		
		anRGBAColor[i]			= theDarkenedColor[i];
	}
	anRGBAColor[3] = 1.0;	//	fully opaque
}
